#!/bin/sh
# Copyright (c) 2014 Synology Inc. All rights reserved.

SURVEILLANCE_INC_SCRP=/var/packages/SurveillanceStation/target/scripts/SurveillanceStationCommon.sh
. ${SURVEILLANCE_INC_SCRP}

# log related table are moved into log.db now
OLD_TABLE_LIST_IN_REC_DB="^event$ ^event_ds_.* ^analyticsevent$ ^alertevent$ ^log$ ^log_ds_.* ^message$ ^edgevideoclip$ ^snapshot$ ^snapshot_ds_.* ^actrulehistory$ ^customaudiopattern"

ReplaceSqlSymbolForSqlite()
{
	local SrcSqlFile="$1"

	if [ ! -f "${SrcSqlFile}" ]; then
		SSDebugLog "Wrong sql script file: ${SrcSqlFile}."
		exit 1
	fi

	echo "BEGIN;"
	cat "${SrcSqlFile}" | \
		sed "s/camid integer NOT NULL/camid integer PRIMARY KEY/g" | \
		sed "s/cameraid integer NOT NULL/cameraid integer PRIMARY KEY/g" | \
		sed "s/id integer NOT NULL/id integer PRIMARY KEY AUTOINCREMENT/g" | \
		sed "s/serial unique/integer PRIMARY KEY AUTOINCREMENT/g" | \
		sed "s/false/0/g" | \
		sed "s/true/1/g" | \
		sed "s/::character varying//g" | \
		sed "s/::text//g" | \
		sed "s/public.//g"
	echo "COMMIT;"
}

AddBusyTimeoutForSqliteFile()
{
	local BusyTimeout="$1"
	local SrcSqlFile="$2"

	if [ ! -f "${SrcSqlFile}" ]; then
		SSDebugLog "Wrong sql script file: ${SrcSqlFile}."
		exit 1
	fi

	echo ".timeout " ${BusyTimeout}
	cat "${SrcSqlFile}"
}

# usage: DropSqliteDBUnnecessaryTables DBPath IsSystemDB
# options:
#	IsSystemDB: [true|false]
DropSqliteDBUnnecessaryTables()
{
	local DBFile="$1"
	local blSystemDB="$2"
	local TableList=$(Sqlite3Exec "${DBFile}" "SELECT name FROM sqlite_master WHERE type='table';")

	for Table in ${TableList}; do
		local blDropTable
		if [ ${blSystemDB} == true ]; then
			blDropTable=false
		else
			blDropTable=true
		fi

		for TableInRecDB in ${OLD_TABLE_LIST_IN_REC_DB}; do
			local TableExists=$(echo ${Table} | grep -i "${TableInRecDB}")

			if [ -n "${TableExists}" ]; then
				if [ ${blSystemDB} == true ]; then
					blDropTable=true
				else
					blDropTable=false
				fi
				break
			fi
		done

		if [ ${blDropTable} == true ]; then
			Sqlite3Exec "${DBFile}" "DROP TABLE ${Table};"
		fi
	done
}

DumpSqliteData()
{
	local DbPath="$1"
	local Tables=$(echo "$2" | sed 's/,/\\|/g')

	if [ ! -f "${DbPath}" ]; then
		SSDebugLog "Wrong db file: ${DbPath}."
		exit 1
	fi

	if [ -z "${Tables}" ]; then
		Sqlite3Exec "${DbPath}" ".dump" | grep -i '^INSERT INTO' | grep -vi '^INSERT INTO "sqlite_sequence"'
	else
		Sqlite3Exec "${DbPath}" ".dump" | grep -i '^INSERT INTO "\('"${Tables}"'\)"'
	fi
}

FixMalformDb()
{
	local Db="$1"

	if [ ! -f "$Db" ]; then
		SSDebugLog "Database file [$Db] does not exist."
		exit 1
	fi

	local BakDbPath="$SS_SERVICE_DATA_FOLDER_LINK/$(basename $Db).malform"
	local DumpDbPath=${Db}.dump

	if [ -d $SS_SERVICE_DATA_FOLDER_LINK/ ]; then
		cp $Db $BakDbPath
		SSDebugLog "Backup malform db to '$BakDbPath'"
	fi

	if rm -f $DumpDbPath && \
		Sqlite3Exec $Db ".clone $DumpDbPath" && \
		Sqlite3Exec $Db ".restore $DumpDbPath"; then
		SSDebugLog "Malform db [$Db] has been fixed."
	else
		SSDebugLog "Failed to fix malform db [$Db]"
		RebuildDbByReindex "$Db"
	fi

	rm -f $DumpDbPath
}

RebuildDbByReindex()
{
	local Db="$1"
	local DbName="$(basename $Db)"

	if [ "$SS_REC_DB_NAME" == "$DbName" ]; then
		RecreateDb "${SS_REC_DB}" "${SS_REC_SQL_SCRIPT}"
		${SS_INDEXER} -n -r -1 -a -1
		return
	fi

	if [ "$SS_REC_CNT_DB_NAME" == "$DbName" ]; then
		RecreateDb "${SS_REC_CNT_DB}" "${SS_REC_CNT_SQL_SCRIPT}"
		${S82_SCRIPT} trigger-ssctl rebuild-recording-cnt
		return
	fi

	if [ "$SS_SNAPSHOT_DB_NAME" == "$DbName" ]; then
		RecreateDb "${SS_SNAPSHOT_DB}" "${SS_SNAPSHOT_SQL_SCRIPT}"
		${SS_INDEXER} -n -s
		return
	fi
}

Usage()
{
	cat << EOF
Usage: $(basename $0) [OPTION]
	--replace-sql-symbol-for-sqlite SQL_FILE RESULT_FILE
	--drop-sqlite-unnecessary-tables SQLITE_DB_FILE IS_SYSTEM_DB
		IS_SYSTEM_DB: [true|false]
	--dump-sqlite-data SQLITE_DB_FILE RESULT_FILE TABLES
		TABLES: [table,table,...]
	--fix-malform-database SQLITE_DB_FILE
	--prepare-service-data-dir
EOF
}

main()
{
	case $1 in
	--replace-sql-symbol-for-sqlite)
		ReplaceSqlSymbolForSqlite "$2" > "$3"
		;;
	--add-busy-timeout-for-sqlite-file)
		AddBusyTimeoutForSqliteFile "$2" "$3" > "$4"
		;;
	--drop-sqlite-unnecessary-tables)
		DropSqliteDBUnnecessaryTables "$2" "$3"
		;;
	--dump-sqlite-data)
		DumpSqliteData "$2" "$4" >> "$3"
		;;
	--fix-malform-database)
		FixMalformDb "$2"
		;;
	--prepare-service-data-dir)
		PrepareServiceDataDir
		;;
	*)
		Usage
		;;
	esac
}

main "$@"
